--
-- This file contains both Oracle and SQL Server sql statements that
-- illustrate the differences between Oracle and SQL Server replication.
-- All of the code in this file is taken directly from Chapter 13 of 
-- Oracle8i and SQL Server 2000 Integration.
--

--
-- Code to illustrate Oracle replication using MATERIALIZED VIEWS
-- 

CREATE TABLE BIKE (
  BIKE_NO NUMBER NOT NULL,
  BIKE_MANUFACTURER VARCHAR2(30) NOT NULL,
  BIKE_NAME         VARCHAR2(50) NOT NULL
)
TABLESPACE USERS 
STORAGE (INITIAL 1M NEXT 1M PCTINCREASE 0)
;

ALTER TABLE BIKE ADD CONSTRAINT PK_BIKE
PRIMARY KEY(BIKE_NO)
USING INDEX TABLESPACE USERS
STORAGE (INITIAL 200K NEXT 200K PCTINCREASE 0)
;

INSERT INTO BIKE VALUES (1,'TREK','Y33')
;

INSERT INTO BIKE VALUES (2,'ROCKY MOUNTAIN','RM6')
;

COMMIT
;

ANALYZE TABLE BIKE COMPUTE STATISTICS
;

CREATE MATERIALIZED VIEW LOG ON BIKE
TABLESPACE USERS 
STORAGE (INITIAL 1M NEXT 1M PCTINCREASE 0)
WITH PRIMARY KEY
;

CREATE MATERIALIZED VIEW BIKE
STORAGE (INITIAL 1M NEXT 1M PCTINCREASE 0)
TABLESPACE USERS
BUILD IMMEDIATE
REFRESH FORCE
WITH PRIMARY KEY
START WITH to_date('01-NOV-2000 08:00:00','DD-MON-YYYY HH24:MI:SS')
 NEXT SYSDATE + 1/24
 AS SELECT * FROM BIKE@PROD
;

INSERT INTO BIKE VALUES (3,'SCHWINN','MOAB 2')
;

INSERT INTO BIKE VALUES (4,'GT','ZASKAR')
;

execute DBMS_REFRESH.REFRESH('"SCOTT"."BIKE"')
;

select job, to_char(next_date,'YYYY-MM-DD HH24:MM:SS') next, what 
from user_jobs
;

--
-- Code to illustrate SQL Server replication
-- 

use master
go

exec sp_replicationdboption @dbname = N'Northwind', 
                            @optname = N'publish', 
                            @value = N'true'
go

use [Northwind]
go

exec sp_addpublication @publication = N'Northwind', 
                       @restricted = N'false', 
                       @sync_method = N'native', 
                       @repl_freq = N'snapshot', 
                       @description = N'Snapshot publication.', 
                       @status = N'active', 
                       @allow_push = N'true', 
                       @allow_pull = N'true', 
                       @allow_anonymous = N'false', 
                       @enabled_for_internet = N'false', 
                       @independent_agent = N'false', 
                       @immediate_sync = N'false', 
                       @allow_sync_tran = N'false', 
                       @autogen_sync_procs = N'false', 
                       @retention = 72
go

exec sp_addpublication_snapshot 
                       @publication = N'Northwind',
                       @frequency_type = 4, 
                       @frequency_interval = 1, 
                       @frequency_relative_interval = 0, 
                       @frequency_recurrence_factor = 1, 
                       @frequency_subday = 1, 
                       @frequency_subday_interval = 0, 
                       @active_start_date = 0, 
                       @active_end_date = 0, 
                       @active_start_time_of_day = 233000, 
                       @active_end_time_of_day = 0
go

exec sp_grant_publication_access @publication = N'Northwind', 
                                 @login = N'BUILTIN\Administrators'
go

exec sp_grant_publication_access @publication = N'Northwind', 
                                 @login = N'distributor_admin'
go

exec sp_grant_publication_access @publication = N'Northwind', 
                                 @login = N'sa'
go

-- Adding the transactional articles

exec sp_addarticle @publication = N'Northwind', 
                   @article = N'Products', 
                   @source_owner = N'dbo', 
                   @source_object = N'Products', 
                   @destination_table = N'Products', 
                   @type = N'logbased', 
                   @creation_script = null, 
                   @description = null, 
                   @pre_creation_cmd = N'drop', 
                   @schema_option = 0x00000000000000F1, 
                   @destination_owner = N'dbo', 
                   @status = 0, 
                   @vertical_partition = N'false', 
                   @ins_cmd = N'SQL',
                   @del_cmd = N'SQL', 
                   @upd_cmd = N'SQL', 
                   @filter = null, 
                   @sync_object = null
go

use warehouse
go

exec sp_addpullsubscription @publisher = N'PROD_DB', 
                            @publisher_db = N'Northwind',
                            @publication = N'Northwind', 
                            @independent_agent = N'false', 
                            @subscription_type = N'pull', 
                            @description = N'Snapshot publication', 
                            @update_mode = N'read only', 
                            @immediate_sync = 0
go


exec sp_addpullsubscription_agent 
                            @publisher = N'PROD_DB', 
                            @publisher_db = N'Northwind', 
                            @publication = N'Northwind', 
                            @distributor = N'PROD_DB', 
                            @subscriber_security_mode = 1, 
                            @distributor_security_mode = 0, 
                            @distributor_login = N'sa', 
                            @distributor_password = N'', 
                            @frequency_type = 64, 
                            @frequency_interval = 1, 
                            @frequency_relative_interval = 1, 
                            @frequency_recurrence_factor = 0, 
                            @frequency_subday = 4, 
                            @frequency_subday_interval = 5, 
                            @active_start_date = 0, 
                            @active_end_date = 0, 
                            @active_start_time_of_day = 0, 
                            @active_end_time_of_day = 235959, 
                            @enabled_for_syncmgr = N'false', 
                            @use_ftp = N'false', 
                            @publication_type = 1, 
                            @dts_package_location = N'subscriber', 
                            @offloadagent = N'false'
go

--
-- Script to be run at Publisher: PROD_DB
--

use [Northwind]
go

exec sp_addsubscription @publication = N'Northwind', 
                        @subscriber = N'DEV_DB', 
                        @destination_db = N'warehouse', 
                        @sync_type = N'automatic', 
                        @subscription_type = N'pull', 
                        @update_mode = N'read only'
go


--
-- Examples showing the creation of Materialized Views/Snapshots
--


GRANT CREATE MATERIALIZED VIEW TO SCOTT
;

CREATE SNAPSHOT "SCOTT"."EMPLOYEES"
BUILD IMMEDIATE
USING INDEX TABLESPACE "INDX"
REFRESH COMPLETE WITH ROWID
START WITH to_date('01-Oct-2000 06:05 PM','DD-MON-YYYY HH12:MI PM')
NEXT sysdate + 1
AS
SELECT "EmployeeID",
       "LastName",
       "FirstName",
       "Title",
       "TitleOfCourtesy",
       "BirthDate",
       "HireDate",
       "Address",
       "City",
       "Region",
       "PostalCode",
       "Country",
       "HomePhone",
       "Extension",
       "ReportsTo",
       "PhotoPath"
  FROM "Employees"@MSOLEDB
;


--
-- Example showing how to use triggers to simulate replication from a SQL
-- Server database to Oracle via a linked server
--

--
-- Oracle CREATE TABLE statement
--

CREATE TABLE SCOTT.BUSINESS_UNITS (
 BU_NAME        VARCHAR2(30)  not null,
 BU_DESCRIPTION VARCHAR2(200) not null
)
TABLESPACE USERS
STORAGE (INITIAL 8K NEXT 8K PCTINCREASE 0)
;

ALTER TABLE SCOTT.BUSINESS_UNIT 
ADD CONSTRAINT BUSINESS_UNIT_PK PRIMARY_KEY (BU_NAME)
USING INDEX TABLESPACE INDX
STORAGE (INITIAL 8K NEXT 8K PCTINCREASE 0)
;

--
-- SQL Server CREATE TABLE statement
--

CREATE TABLE BUSINESS_UNIT (
 BU_NAME        VARCHAR(30)  not null,
 BU_DESCRIPTION VARCHAR(200) not null
)
go


-- 
-- Triggers created on SQL Server database using the ORACLE8I linked server
-- to access the BUSINESS_UNIT table in Oracle
--

CREATE TRIGGER BUSINESS_UNIT_INSERT ON dbo.BUSINESS_UNIT
FOR INSERT 
AS
BEGIN
  INSERT INTO "ORACLE8I".."SCOTT"."BUSINESS_UNIT"
    (BU_NAME, BU_DESCRIPTION)
  SELECT inserted.BU_NAME, inserted.BU_DESCRIPTION
  FROM inserted
END
go

CREATE TRIGGER BUSINESS_UNIT_DELETE ON dbo.BUSINESS_UNIT
FOR DELETE 
AS
BEGIN
  DELETE FROM "ORACLE8I".."SCOTT"."BUSINESS_UNIT"
  WHERE BU_NAME = 
      ( SELECT deleted.BU_NAME FROM deleted)
END
go

CREATE TRIGGER BUSINESS_UNIT_UPDATE ON dbo.BUSINESS_UNIT
FOR UPDATE 
AS
BEGIN
  DECLARE
    @old_bu_name nvarchar(30),
    @old_bu_description nvarchar(200),
    @new_bu_name nvarchar(30),
    @new_bu_description nvarchar(200)
  
  SELECT @old_bu_name = deleted.BU_NAME,
         @old_bu_description = deleted.BU_DESCRIPTION
  FROM deleted

  SELECT @new_bu_name = inserted.BU_NAME,
         @new_bu_description = inserted.BU_DESCRIPTION
  FROM inserted

  UPDATE "ORACLE8I".."SCOTT"."BUSINESS_UNIT"
  SET BU_NAME = @new_bu_name, 
      BU_DESCRIPTION = @new_bu_description
  WHERE BU_NAME = @old_bu_name
END
go


--
-- Sample INSERT in Transact-SQL which will cause the same data to be inserted
-- in the Oracle database as a single transaction
--

SET XACT_ABORT ON

BEGIN TRANSACTION
  INSERT INTO BUSINESS_UNIT (BU_NAME, BU_DESCRIPTION)
  VALUES('Test Unit', 'This is a test')
COMMIT TRANSACTION
go




--
-- Example showing how to use triggers to simulate replication from an
-- Oracle database to a SQL Server database using Oracle Heterogeneous
-- Services
--

INSERT INTO SCOTT.EMPLOYEES(EMPLOYEEID, LASTNAME, FIRSTNAME, TITLE, 
   TITLEOFCOURTESY, BIRTHDATE, HIREDATE, ADDRESS, CITY, REGION,
   POSTALCODE, COUNTRY, HOMEPHONE, EXTENSION, REPORTSTO, PHOTOPATH)
SELECT "EmployeeID",
       "LastName",
       "FirstName",
       "Title",
       "TitleOfCourtesy",
       "BirthDate",
       "HireDate",
       "Address",
       "City",
       "Region",
       "PostalCode",
       "Country",
       "HomePhone",
       "Extension",
       "ReportsTo",
       "PhotoPath"
  FROM "Employees"@MSOLEDB
;

CREATE TABLE SCOTT.EMPLOYEES 
TABLESPACE DATA_TABLESPACE 
STORAGE(INITIAL 100K NEXT 100K PCTINCREASE 0) AS
SELECT "EmployeeID",
       "LastName",
       "FirstName",
       "Title",
       "TitleOfCourtesy",
       "BirthDate",
       "HireDate",
       "Address",
       "City",
       "Region",
       "PostalCode",
       "Country",
       "HomePhone",
       "Extension",
       "ReportsTo",
       "PhotoPath"
  FROM "Employees"@MSOLEDB
;


CREATE TABLE HS_TRANSACTION_LOG (
  GLOBAL_TRAN_ID  CHAR(64)  NOT NULL,
  TRAN_COMMENT    CHAR(255) NOT NULL
)
go


CREATE TRIGGER SCOTT.BUSINESS_UNIT_BIR
BEFORE INSERT ON SCOTT.BUSINESS_UNIT
FOR EACH ROW BEGIN
  INSERT INTO "dbo"."BUSINESS_UNIT"@SQLGATEWAY (
    "BU_NAME",
    "BU_DESCRIPTION"
  )
  VALUES (
    :new.BU_NAME,
    :new.BU_DESCRIPTION);
END;
/

CREATE TRIGGER SCOTT.BUSINESS_UNIT_BUR 
BEFORE UPDATE OF BU_DESCRIPTION, BU_NAME 
ON SCOTT.BUSINESS_UNIT
FOR EACH ROW BEGIN
  UPDATE "dbo"."BUSINESS_UNIT"@SQLGATEWAY
     SET "BU_NAME" = :new.BU_NAME,
         "BU_DESCRIPTION" = :new.BU_DESCRIPTION
   WHERE "BU_NAME" = :old.BU_NAME;
END;
/

CREATE TRIGGER SCOTT.BUSINESS_UNIT_BDR 
BEFORE DELETE ON SCOTT.BUSINESS_UNIT
FOR EACH ROW BEGIN
  DELETE "dbo"."BUSINESS_UNIT"@SQLGATEWAY
  WHERE "BU_NAME" = :old.BU_NAME;
END;
/

-- 
-- How to force information to be replicated from the SCOTT.EMP table using
-- the ORACLE8I linked server
--

SELECT * 
INTO EMP
FROM ORACLE8I..SCOTT.EMP
go


--
-- or you can use
--

TRUNCATE TABLE EMP
go

INSERT INTO EMP
SELECT * FROM ORACLE8I..SCOTT.EMP
go

